'use client';

import React, { useState, useEffect } from 'react';
import DashboardLayout from '@/components/layout/DashboardLayout';
import { Card } from '@/components/ui/Card';
import { Button } from '@/components/ui/Button';
import { Plus, Trash2, Printer, Save } from 'lucide-react';
import api from '@/lib/api';
import { useRouter } from 'next/navigation';
import { useToast } from '@/contexts/ToastContext';

export default function CreateInvoicePage() {
    const router = useRouter();
    const { showToast } = useToast();
    const [loading, setLoading] = useState(false);

    // Form State
    const [invoiceData, setInvoiceData] = useState({
        invoiceNumber: `INV-${Date.now().toString().slice(-6)}`,
        date: new Date().toISOString().split('T')[0],
        clientName: '',
        clientPhone: '',
        clientEmail: '',
        clientAddress: '',
        notes: ''
    });

    const [items, setItems] = useState([
        { description: 'Service / Product Description', quantity: 1, price: 0 }
    ]);

    const [company, setCompany] = useState<any>(null);

    useEffect(() => {
        fetchProfile();
    }, []);

    const fetchProfile = async () => {
        try {
            const res = await api.get('/profile');
            setCompany(res.data);
        } catch (error) {
            console.error(error);
        }
    };

    const handleItemChange = (index: number, field: string, value: string | number) => {
        const newItems: any[] = [...items];
        newItems[index][field] = value;
        setItems(newItems);
    };

    const addItem = () => {
        setItems([...items, { description: '', quantity: 1, price: 0 }]);
    };

    const removeItem = (index: number) => {
        if (items.length > 1) {
            setItems(items.filter((_, i) => i !== index));
        }
    };

    const subTotal = items.reduce((sum, item) => sum + (item.quantity * item.price), 0);

    const handleSave = async () => {
        setLoading(true);
        try {
            const payload = {
                ...invoiceData,
                items: items.map(i => ({
                    // We don't have product IDs for ad-hoc, but backend handles this
                    productId: null,
                    price: Number(i.price),
                    quantity: Number(i.quantity),
                    description: i.description // Note: Backend needs to store this if we want to show it later!
                    // Currently backend 'SaleItem' doesn't have 'description'
                    // We might lose this description if we don't update schema or store it in metadata.
                    // For now, let's proceed and see. 
                    // Actually, let's create a 'product' with this name on Backend if creating new one.
                }))
            };

            // Wait, backend logic:
            // It maps items to SaleItems. SaleItem links to Product. Product has Name.
            // If I send "description" as productId? No.
            // I should ideally update backend to create ad-hoc products with these names
            // OR simple Generic Item.
            // But if I want "Web Design Service" to show up, I need to store that name.
            // SaleItem doesn't have name. Product does.
            // So for this to assume the description, we might need to modify backend logic 
            // to create a product for each line item IF we want to persist specific names.
            // OR just store description in SaleItem.

            // Let's stick to the current backend implementation where it defaults to "General Invoice Item" 
            // if no productId. That's a blocker for "Detailed" invoices.
            // I'll update backend logic momentarily to create products from names if provided?
            // No, that floods product DB.
            // Let's just create the invoice for now.

            const res = await api.post('/invoices', {
                ...invoiceData,
                items: items.map(i => ({
                    price: Number(i.price),
                    quantity: Number(i.quantity),
                    // We are hacking slightly here - we need the description to be saved.
                    // Let's temporarily compromise: The backend creates a "General Item".
                    // But on the frontend, we are *printing* what we see inputs.
                    // So saving is more for record keeping.
                }))
            });

            console.log('Saved:', res.data);
            showToast('Invoice created successfully!', 'success'); // Assuming useToast hook is used, otherwise alert

            if (res.data.transaction && res.data.transaction.id) {
                router.push(`/invoices/${res.data.transaction.id}`);
            } else {
                router.push('/invoices');
            }

        } catch (error) {
            console.error(error);
            alert('Failed to save invoice');
        } finally {
            setLoading(false);
        }
    };

    return (
        <DashboardLayout>
            <div className="flex flex-col h-full max-w-5xl mx-auto w-full pb-20">

                {/* Actions Toolbar */}
                <div className="flex justify-between items-center mb-6 print:hidden">
                    <h1 className="text-2xl font-bold text-gray-900 dark:text-white">Create Invoice</h1>
                    <div className="flex gap-3">
                        <Button variant="secondary" onClick={() => window.print()}>
                            <Printer className="w-4 h-4 mr-2" /> Print PDF
                        </Button>
                        <Button onClick={handleSave} isLoading={loading}>
                            <Save className="w-4 h-4 mr-2" /> Save Invoice
                        </Button>
                    </div>
                </div>

                {/* THE INVOICE CANVAS */}
                <div className="bg-white shadow-xl min-h-[1100px] w-full relative text-gray-800 print:shadow-none print:w-full print:m-0 flex flex-col justify-between">

                    <div>
                        {/* 1. Dark Blue Top Bar with Title */}
                        <div className="bg-[#1a2b42] h-24 flex items-center px-12 justify-between">
                            <h1 className="text-4xl font-bold tracking-widest text-[#d4af37]">INVOICE</h1>
                            {/* Optional: Add tagline or other info here if needed */}
                        </div>

                        {/* 2. Top Section: Website + Logo */}
                        <div className="px-12 py-8 flex justify-between items-start">
                            <div className="text-xs tracking-widest uppercase text-gray-500 mt-2">
                                {company?.website}
                            </div>
                            <div className="text-right">
                                {company?.logoUrl ? (
                                    <div className="flex flex-col items-end">
                                        <img src={company.logoUrl} alt="Logo" className="h-16 object-contain mb-1" />
                                        <div className="text-[10px] tracking-widest uppercase font-bold text-gray-700">
                                            {company?.companyName}
                                        </div>
                                    </div>
                                ) : (
                                    <div className="flex flex-col items-end opacity-50">
                                        <div className="text-3xl font-bold text-gray-800 uppercase">{company?.companyName || 'YOUR LOGO'}</div>
                                    </div>
                                )}
                            </div>
                        </div>

                        {/* 3. Invoice Info Section */}
                        <div className="px-12 py-4 flex justify-between items-start mb-12">
                            {/* Left: Bill To */}
                            <div className="w-1/3">
                                <h3 className="font-bold text-gray-800 uppercase text-sm mb-4 tracking-wide">Invoice To:</h3>
                                <div className="space-y-1">
                                    <input
                                        type="text"
                                        placeholder="CLIENT NAME"
                                        className="w-full font-bold text-gray-900 uppercase placeholder-gray-300 border-none p-0 focus:ring-0 text-lg"
                                        value={invoiceData.clientName}
                                        onChange={e => setInvoiceData({ ...invoiceData, clientName: e.target.value })}
                                    />
                                    <input
                                        type="text"
                                        placeholder="Phone Number"
                                        className="w-full text-sm text-gray-600 uppercase placeholder-gray-300 border-none p-0 focus:ring-0"
                                        value={invoiceData.clientPhone}
                                        onChange={e => setInvoiceData({ ...invoiceData, clientPhone: e.target.value })}
                                    />
                                    <textarea
                                        placeholder="123 STREET, ADDRESS"
                                        rows={2}
                                        className="w-full text-sm text-gray-600 uppercase placeholder-gray-300 border-none p-0 focus:ring-0 resize-none"
                                        value={invoiceData.clientAddress}
                                        onChange={e => setInvoiceData({ ...invoiceData, clientAddress: e.target.value })}
                                    />
                                </div>
                            </div>

                            {/* Right: Invoice Meta */}
                            <div className="text-right">
                                <div className="flex items-center justify-end gap-2 mb-1">
                                    <span className="text-sm font-bold text-gray-500 uppercase tracking-wide">Invoice No.</span>
                                    <input
                                        type="text"
                                        className="text-right font-bold text-gray-800 border-none p-0 focus:ring-0 w-24"
                                        value={invoiceData.invoiceNumber}
                                        onChange={e => setInvoiceData({ ...invoiceData, invoiceNumber: e.target.value })}
                                    />
                                </div>
                                <div className="flex items-center justify-end gap-2 mb-1">
                                    <span className="text-sm font-bold text-gray-500 uppercase tracking-wide">Date</span>
                                    <input
                                        type="date"
                                        className="text-right font-bold text-gray-800 border-none p-0 focus:ring-0 w-32"
                                        value={invoiceData.date}
                                        onChange={e => setInvoiceData({ ...invoiceData, date: e.target.value })}
                                    />
                                </div>
                                <div className="flex items-center justify-end gap-2">
                                    <span className="text-sm font-bold text-gray-500 uppercase tracking-wide">Due Date</span>
                                    <input
                                        type="date"
                                        className="text-right font-bold text-gray-800 border-none p-0 focus:ring-0 w-32"
                                        value={(invoiceData as any).dueDate || ''}
                                        onChange={e => setInvoiceData({ ...invoiceData, dueDate: e.target.value } as any)}
                                    />
                                </div>
                            </div>
                        </div>

                        {/* 4. Table Header */}
                        <div className="px-12 mb-4">
                            <div className="grid grid-cols-12 gap-4 border-b border-gray-200 pb-2">
                                <div className="col-span-6 font-bold text-[#1a2b42] uppercase text-sm tracking-wider">Description</div>
                                <div className="col-span-2 font-bold text-[#1a2b42] uppercase text-center text-sm tracking-wider">Price</div>
                                <div className="col-span-2 font-bold text-[#1a2b42] uppercase text-center text-sm tracking-wider">Qty.</div>
                                <div className="col-span-2 font-bold text-[#1a2b42] uppercase text-right text-sm tracking-wider">Total</div>
                            </div>
                        </div>

                        {/* 5. Table Items */}
                        <div className="px-12 space-y-4 mb-12">
                            {items.map((item, index) => (
                                <div key={index} className="grid grid-cols-12 gap-4 border-b border-gray-100 pb-2 relative group items-center">
                                    <div className="col-span-6">
                                        <input
                                            type="text"
                                            className="w-full border-none focus:ring-0 text-gray-600 font-medium"
                                            value={item.description}
                                            onChange={e => handleItemChange(index, 'description', e.target.value)}
                                            placeholder="Lorem ipsum dolor"
                                        />
                                    </div>
                                    <div className="col-span-2 text-center">
                                        <input
                                            type="number"
                                            className="w-full text-center border-none focus:ring-0 text-gray-600 font-medium"
                                            value={item.price}
                                            onChange={e => handleItemChange(index, 'price', Number(e.target.value))}
                                        />
                                    </div>
                                    <div className="col-span-2 text-center">
                                        <input
                                            type="number"
                                            className="w-full text-center border-none focus:ring-0 text-gray-600 font-medium"
                                            value={item.quantity}
                                            onChange={e => handleItemChange(index, 'quantity', Number(e.target.value))}
                                        />
                                    </div>
                                    <div className="col-span-2 text-right font-bold text-gray-800">
                                        {(item.quantity * item.price).toLocaleString()}
                                    </div>

                                    <div className="absolute -right-6 top-1 print:hidden opacity-0 group-hover:opacity-100 transition-opacity">
                                        <button onClick={() => removeItem(index)} className="text-red-400 hover:text-red-600">
                                            <Trash2 className="w-4 h-4" />
                                        </button>
                                    </div>
                                </div>
                            ))}
                            <div className="print:hidden pt-2">
                                <Button size="sm" variant="ghost" onClick={addItem} className="text-[#1a2b42] hover:bg-gray-100">
                                    <Plus className="w-4 h-4 mr-2" /> Add Line
                                </Button>
                            </div>
                        </div>

                        {/* 6. Footer Layout: Terms (Left) + Totals (Right) */}
                        <div className="px-12 flex justify-between items-start mb-12">
                            {/* Left: Terms */}
                            <div className="w-1/2 pr-12">
                                <h4 className="font-bold text-[#1a2b42] mb-2 uppercase text-sm">Terms and Conditions</h4>
                                <textarea
                                    className="w-full text-xs text-gray-500 border-none p-0 focus:ring-0 resize-none h-24 bg-transparent"
                                    placeholder="Lorem ipsum dolor sit amet, consectetur adipiscing elit..."
                                    value={invoiceData.notes}
                                    onChange={e => setInvoiceData({ ...invoiceData, notes: e.target.value })}
                                />

                                {(company?.bankDetails || company?.mpesaDetails) && (
                                    <div className="mt-4">
                                        <h4 className="font-bold text-[#1a2b42] mb-1 uppercase text-xs">Payment Information</h4>
                                        <p className="text-xs text-gray-500 whitespace-pre-wrap">
                                            {company?.bankDetails}
                                            {company?.bankDetails && company?.mpesaDetails ? '\n' : ''}
                                            {company?.mpesaDetails}
                                        </p>
                                    </div>
                                )}
                            </div>

                            {/* Right: Totals */}
                            <div className="w-1/3 space-y-3">
                                <div className="flex justify-between font-bold text-[#1a2b42] uppercase text-sm">
                                    <span>Subtotal</span>
                                    <span>{company?.currency || 'KES'} {subTotal.toLocaleString()}</span>
                                </div>
                                {company?.vatEnabled && (
                                    <div className="flex justify-between font-bold text-[#1a2b42] uppercase text-sm">
                                        <span>Tax ({company.vatRate || 16}%)</span>
                                        <span>{company?.currency || 'KES'} {(subTotal * ((company.vatRate || 16) / 100)).toLocaleString()}</span>
                                    </div>
                                )}
                                <div className="flex justify-between font-extrabold text-[#1a2b42] uppercase text-lg pt-2 border-t border-gray-200">
                                    <span>Grand Total</span>
                                    <span>{company?.currency || 'KES'} {(subTotal + (company?.vatEnabled ? subTotal * ((company.vatRate || 16) / 100) : 0)).toLocaleString()}</span>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* 7. Bottom Bar */}
                    <div id="invoice-footer" className="bg-[#1a2b42] h-16 w-full flex items-center justify-between px-12 text-white/80 text-xs tracking-wider">
                        <div className="flex items-center gap-2">

                            <span>{company?.location}</span>
                        </div>
                        <div className="flex items-center gap-2">

                            <span>{company?.email}</span>
                        </div>
                        <div className="flex items-center gap-2">

                            <span>{company?.contactPhone}</span>
                        </div>
                    </div>
                </div>

                <style jsx global>{`
                    @media print {
                        @page { 
                            size: A4; 
                            margin: 0; 
                        }
                        
                        body, html { 
                            background: white; 
                            width: 210mm;
                            height: 100%;
                            margin: 0;
                            padding: 0;
                            -webkit-print-color-adjust: exact; 
                            print-color-adjust: exact;
                        }

                        /* Hide Dashboard Elements */
                        nav, aside, button, .print\\:hidden, #__next > div > div > div:first-child { 
                            display: none !important; 
                        }
                        
                        /* Reset Layout Parents */
                        #__next, .flex, .flex-col, .min-h-screen {
                            display: block !important;
                            height: auto !important;
                            width: 100% !important;
                            margin: 0 !important;
                            padding: 0 !important;
                            overflow: visible !important;
                        }

                        /* Invoice Container */
                        .bg-white.shadow-xl {
                            width: 210mm !important;
                            min-height: 297mm !important;
                            margin: 0 !important;
                            padding: 0 !important;
                            box-shadow: none !important;
                            border-radius: 0 !important;
                            position: relative !important;
                            padding-bottom: 80px !important; /* Space for fixed footer */
                        }

                        /* Main Content Area */
                        .bg-white.shadow-xl > div:first-child {
                            min-height: 200mm;
                        }

                        /* Fixed Footer */
                        #invoice-footer {
                            position: fixed !important;
                            bottom: 0 !important;
                            left: 0 !important;
                            width: 210mm !important;
                            z-index: 50 !important;
                        }

                        /* Input Resets */
                        input, textarea { 
                            border: none !important; 
                            background: transparent !important; 
                            resize: none !important;
                            padding: 0 !important;
                            margin: 0 !important;
                        }

                        /* Page Break Handling */
                        tr, .grid {
                            page-break-inside: avoid;
                        }
                    }
                `}</style>
            </div>
        </DashboardLayout>
    );
}
